/*++
This file contains a 'Sample Driver' and is licensed as such
under the terms of your license agreement with Intel or your
vendor.  This file may be modified by the user, subject to  
the additional terms of the license agreement               
--*/
/*++

Copyright (c)  2009-2017 Intel Corporation. All rights reserved
This software and associated documentation (if any) is furnished
under a license and may only be used or copied in accordance
with the terms of the license. Except as permitted by such
license, no part of this software or documentation may be
reproduced, stored in a retrieval system, or transmitted in any
form or by any means without the express written consent of
Intel Corporation.

Module Name:

FwUpdLclApp.c

Abstract:

FW Update Local Sample Code demonstrating usage of FW Update Library

--*/

#include "efi.h"
#include "efilib.h"
#include "Fwu_Common.h"
#include "Common.h"
#include "me_status.h"
#include "FWUpdateLib.h"
#include "cse_basic_types.h"
#include "cse_nvm_defs.h"
#include "typedef.h"
#include "types.h"
#include "tool_version.h"

#define CMD_LINE_STATUS_UPDATE_1        0
#define CMD_LINE_STATUS_UPDATE_2        1
#define CMD_LINE_STATUS_UPDATE_3        2
#define CMD_LINE_STATUS_UPDATE_4        3

#define HECI1_CSE_GS1_PHASE_FWUPDATE                  7


// Info Strings 
#define ID_INFO_DOT     L"."
#define ID_INFO_1       L"Initiating firmware update process...\n"
#define ID_INFO_2       L"\nTrying to receive update status..."
#define ID_INFO_3       L"Communication Mode: MEI\n"
#define ID_INFO_4       L"Update finished successfully\n"
#define ID_INFO_5       L"\nSending the update image to FW for verification..."
#define ID_INFO_6       L"\nImage successfully sent to FW.\nFW verifying the image..."
#define ID_INFO_7       L"Communication Mode: EOI\n"
#define ID_INFO_8       L"Communication Mode: DASH\n"
#define ID_INFO_9       L"Communication Mode: WSMAN\n"
#define ID_INFO_10      L"\nTrying to connect to MEI driver."
#define ID_INFO_11      L"\nFW Update is complete and a reboot will run the new FW.\n" 
#define ID_INFO_15      L"Stop Configuration finished successfully.\n"
#define ID_INFO_16      L"Maximum Retry limit reached. Please reboot to try again\n"
#define ID_INFO_17      L"Please reboot system or re-attempt FW Update process\n"
#define ID_INFO_18      L"\nFW Update is completed successfully.\n" 
#define ID_INFO_19      L"\nFirmware Downgrade will NOT proceed. Request cancelled by user. \n"
#define ID_INFO_20      L"\nFirmware Update will NOT proceed. Request cancelled by user. \n"
#define ID_INFO_21      L"\nIt is recommended to reboot the platform once the FW update is completed. \n"

// Error Strings
#define ID_ERROR_2      L"\nError %d: Firmware update not initiated due to invalid hostname specified\n"
#define ID_ERROR_3      L"\nError %d: Firmware update process or firmware update tool failed due to insufficient memory\n"
#define ID_ERROR_4      L"\nError %d: Update operation timed-out; cannot determine if the operation succeeded\n"
#define ID_ERROR_5      L"\nError %d: Cannot receive the current version from the firmware after update\n"
#define ID_ERROR_6      L"\nError %d: Update finished but version mismatch after the update\n"
#define ID_ERROR_7      L"\nError %d: Firmware update not initiated due to an invalid FW image \n"
#define ID_ERROR_8      L"\nError %d: Firmware update not initiated due to integrity failure or invalid FW image"
#define ID_ERROR_9      L"\nError %d: Sku capabilities bits are different between the Update Image and the Flash Image.\n"
#define ID_ERROR_10     L"\nError %d: Major version number of Update Image is not the same as major version number of Flash Image.\n"
#define ID_ERROR_11     L"\nError %d: Firmware update failed due to authentication failure\n"
#define ID_ERROR_12     L"\nError %d: Firmware update failed due to insufficient memory\n"
#define ID_ERROR_13     L"\nError %d: Firmware update iAMT communication failed, Failed to find certificate '%s' in certificate store\n"
#define ID_ERROR_14     L"\nError %d: Firmware update iAMT communication failed, Failed to set HTTP certificate options (%d): %s\n"
#define ID_ERROR_15     L"\nError %d: Firmware update iAMT communication failed, Failed to find certificate names\n"
#define ID_ERROR_16     L"\nError %d: Firmware update iAMT communication failed, Failed to open system certificate store (%d): %s\n"
#define ID_ERROR_17     L"\nError %d: Firmware update tool failed to connect iAMT through LMS, due to a HTTP operation failure,\nPlease verify the inputs (host, user, password, certificate, work mode etc).\n"

#define ID_ERROR_18     L"\nError %d: Invalid usage\n"
#define ID_ERROR_18B    L"\nError %d: Invalid usage, -allowsv switch required to update the same version firmware\n"
#define ID_ERROR_19     L"\nError %d: Failed to receive last update status from the firmware\n"
#define ID_ERROR_20     L"\nError %d: Firmware Update operation not initiated because a firmware update is already in progress\n"
#define ID_ERROR_21     L"\nError %d: Invalid UUID provided with the OEMID switch\n"
#define ID_ERROR_22     L"\nError %d: OEM ID verification failed.\n"
#define ID_ERROR_24     L"\nError %d: Firmware update tool failed to get the firmware parameters\n"

#define ID_ERROR_25     L"\nError %d: Firmware update failed due to an internal error\n"
#define ID_ERROR_25A    L"\nError %d: Firmware update failed due to an internal error\nFirmware returns SAL notification error, Please try after ME-reset or re-flashing the ME image.\n"
#define ID_ERROR_25B    L"\nError %d: Firmware update failed due to an internal error\nFirmware returns Audit policy error, Please try after ME-reset or re-flashing the ME image.\n"
#define ID_ERROR_25C    L"\nError %d: Firmware update failed due to an internal error\nThe total size of the backup partitions is bigger than NFTP size. Can happen in Consumer, when not setting fixed partitions sizes in build.\n"
#define ID_ERROR_25D    L"\nError %d: Firmware update failed due to an internal error\nFirmware failed  on image version history check.\n"
#define ID_ERROR_25E    L"\nError %d: Firmware update failed due to an internal error\nWrite file failed: error occurred in write() or number of bytes written is not the same as file length.\n"
#define ID_ERROR_25F    L"\nError %d: Firmware update failed due to an internal error\nRead file failed: error occurred in read() or number of bytes read is not the same as file length.\n"
#define ID_ERROR_25G    L"\nError %d: Firmware update failed due to an internal error\nFirmware failed with delete file issue.\n"
#define ID_ERROR_25H    L"\nError %d: Firmware update failed due to an internal error\nFirmware returns invalid flash code partition.\n"
#define ID_ERROR_25I    L"\nError %d: Firmware update failed due to an internal error\nNFTP is corrupted, CSE is in Recovery Mode.\n"
#define ID_ERROR_25J    L"\nError %d: Firmware update failed due to an internal error\nFirmware returns invalid image length.\n"
#define ID_ERROR_25K    L"\nError %d: Firmware update failed due to an internal error\nFirmware returns invalid GLUT.\n"
#define ID_ERROR_25L    L"\nError %d: Firmware update failed due to an internal error\nFirmware Update client is not ready to perform an update.\n"


#define ID_ERROR_26     L"\nError %d: Unsupported Operating System\n"
#define ID_ERROR_27     L"\nError %d: No Firmware update is happening\n"
#define ID_ERROR_28     L"\nError %d: Unknown or Unsupported Platform\n"
#define ID_ERROR_29     L"\nError %d: FW Update is disabled. MEBX has options to disable / enable / require ME password for FW Update.\n"
#define ID_ERROR_30     L"\nError %d: Firmware update cannot be initiated because Secure FW update is disabled\n"
#define ID_ERROR_31     L"\nError %d: Firmware update not initiated due to an invalid FW image header\n"
#define ID_ERROR_32     L"\nError %d: Firmware update not initiated due to file [%s] open or read failure\n"
#define ID_ERROR_33     L"\nError %d: Firmware update cannot be initiated because the OEM ID given for FW Update did not match the OEM ID in the FW.\n"
#define ID_ERROR_34     L"\nError %d: Firmware update not initiated due to file open or read failure\n"
#define ID_ERROR_35     L"\nError %d: Firmware update iAMT communication failed, HTTP request failed: Certificate rejected.\n"
#define ID_ERROR_36     L"\nError %d: This version of the Intel (R) FW Update Tool is not compatible with the current platform.\n"
#define ID_ERROR_36A    L"\nError %d: Buffer too small\n"
#define ID_ERROR_37     L"\nError %d: Fail to load MEI device driver (PCI access for Windows)\nAbove error is often caused by one of below reasons:\nAdministrator privilege needed for running the tool\nME is in an error state causing MEI driver fail\nMEI driver is not installed\n"
#define ID_ERROR_38     L"\nError %d: Platform did not respond to update request.\n"
#define ID_ERROR_39     L"\nError %d: Intel (R) ME Interface : Unsupported message type\n"
#define ID_ERROR_40     L"\nError %d: Firmware update not initiated. Num of bytes to read/write/erase is bigger than partition size\n"
#define ID_ERROR_41     L"\nError %d: Firmware update not initiated due to an unavailable global buffer\n"
#define ID_ERROR_44     L"\nError %d: Firmware update not initiated due to invalid firmware parameters\n"
#define ID_ERROR_46     L"\nError %d: Firmware update iAMT communication failed, WSMAN not supported\n"
#define ID_ERROR_51     L"\nError %d: PLEASE REBOOT YOUR SYSTEM. Firmware update cannot be initiated without a reboot.\n"
#define ID_ERROR_52     L"\nError %d: An internal error to the AMT device has occurred.\n"
#define ID_ERROR_53     L"\nError %d: AMT Status is not ready.\n"
#define ID_ERROR_54     L"\nError %d: Invalid AMT Mode.\n"
#define ID_ERROR_55     L"\nError %d: Encountered error writing to file.\n"
#define ID_ERROR_56     L"\nError %d: Display FW Version failed.\n"
#define ID_ERROR_57     L"\nError %d: The image provided is not supported by the platform.\n"
#define ID_ERROR_57A    L"\nError %d: This platform does not allow downgrade/upgrade with the file provided.\n"
#define ID_ERROR_58     L"\nError %d: Internal Error.\n"
#define ID_ERROR_59     L"\nError %d: Update downgrade vetoed.\n"
#define ID_ERROR_60     L"\nError %d: Firmware write file failure.\n"
#define ID_ERROR_61     L"\nError %d: Firmware read file failure.\n"
#define ID_ERROR_62     L"\nError %d: Firmware delete file failure.\n"
#define ID_ERROR_63     L"\nError %d: Sanity check in erase/write of partitions. Error might have happened when size of partition is not 4K aligned.\n"
#define ID_ERROR_64     L"\nError %d: Downgrade NOT allowed, data mismatched.\n"
#define ID_ERROR_65     L"\nError %d: The password given for FW Update did not match the correct password (the password is the ME password set in MEBX).\n"
#define ID_ERROR_66     L"\nError %d: The number of attempts to provide password for FW Update exceeds the number of allowed attempts. (3 attempts allowed). (Restart will reset the number of attempts).\n"
#define ID_ERROR_67     L"\nError %d: Password Not provided when required.\n"
#define ID_ERROR_68     L"\nError %d: Polling for FW Update Failed.\n"
#define ID_ERROR_69     L"\nError %d: FW Update Failed.\n"
#define ID_ERROR_70     L"\nError %d: Intel (R) ME Interface : Cannot locate ME device driver; unable to determine if the operation succeeded.\n"
#define ID_ERROR_71     L"\nError %d: Intel (R) ME Interface : Lost communication with platform during update; unable to determine if the update completed.\n"
#define ID_ERROR_72     L"\nError %d: The partition provided is not supported by the platform. Please verify the update image used.\n"
#define ID_ERROR_73     L"\nError %d: Firmware is in recovery mode. Please update only using the previously attempted version.\n"
#define ID_ERROR_74     L"\nError %d: Invalid Partition ID. Use a Partition ID which is possible to do Partial FW Update on.\n"
#define ID_ERROR_75     L"\nError %d: FW update/downgrade is not allowed to the supplied FW image.\n"
#define ID_ERROR_76     L"\nError %d: Update to Image with lower SVN is not allowed.\n"
#define ID_ERROR_77     L"\nError %d: File already exists.\n"
#define ID_ERROR_78     L"\nError %d: Invalid File.\n"
#define ID_ERROR_79     L"\nError %d: Restore Point Image Failure. Reboot may be required. \n"
#define ID_ERROR_80     L"\nError %d: Get Partition Attribute Failure.\n"
#define ID_ERROR_81     L"\nError %d: Get Update Info Status Failure.\n"
#define ID_ERROR_82     L"\nError %d: Buffer Copy Failure.\n"
#define ID_ERROR_83     L"\nError %d: Invalid pointer.\n"
#define ID_ERROR_84     L"\nError %d: UPV version is mismatched.\n"
#define ID_ERROR_85     L"\nError %d: Partial update is allowed only to the expected instance ID of an IUP. The Update Image contains IUP with instance ID that is not the currently expected one by the FW. To update LOCL, please use The Intel Management and Security Status (IMSS) tool. \n"
#define ID_ERROR_86     L"\nError %d: FW info not available. \n"
#define ID_ERROR_87     L"\nError %d: Partial Update is not allowed, because CSE is in Recovery Mode.\n"
#define ID_ERROR_88     L"\nError %d: Partial Update of an IUP was requested, but this IUP doesn't exist in the Flash Image.\n"
#define ID_ERROR_89     L"\nError %d: Update to Restore Point Image: comparing the hash of the image to the saved hashes of last 10 Restore Point Images that were created previously, did not find a match. This means that the image is not a Restore Point Image.\n"
#define ID_ERROR_90     L"\nError %d: Restore point is valid but not the latest. \n"
#define ID_ERROR_91     L"\nError %d: Get Restore Point Image is not allowed, because FW Update is in progress. (The regular FW Update will continue). \n"
#define ID_ERROR_92     L"\nError %d: Update to Image with lower VCN is not allowed. \n"
#define ID_ERROR_93     L"\nError %d: SVN invalid: SVN larger than 254 is not allowed. \n"
#define ID_ERROR_94     L"\nError %d: PSVN partition is full, so cannot update to higher SVN. \n"
#define ID_ERROR_95     L"\nError %d: Restore Point Image was requested, but it is not allowed because CSE is in Recovery Mode. \n"
#define ID_ERROR_96     L"\nError %d: Rejested by update policy. \n"
#define ID_ERROR_97     L"\nError %d: Rejected, incompatible tool usage. \n"
#define ID_ERROR_98     L"\nError %d: Rejected, crosspoint update not allowed. \n"
#define ID_ERROR_99     L"\nError %d: Rejected, crosshotfix update not allowed. \n"
#define ID_ERROR_100    L"\nError %d: Rejected, current fw is not eligible for update. \n"
#define ID_ERROR_101    L"\nError %d: Rejected, wrong update operation. \n"
#define ID_ERROR_102    L"\nError %d: Rejected, wrong update image found. \n"
#define ID_ERROR_103    L"\nError %d: Rejected, IFR update not allowed. \n"
#define ID_ERROR_104    L"\nError %d: Failure occurred during rollback. \n"
#define ID_ERROR_105    L"\nError %d: Restore point operation failed. \n"
#define ID_ERROR_106    L"\nError %d: Invalid message length. \n"
#define ID_ERROR_107    L"\nError %d: Get blist error. \n"
#define ID_ERROR_108    L"\nError %d: Host reset is required after the last FW Update operation. \n"
#define ID_ERROR_109    L"\nError %d: Unknown error code. \n"
#define ID_ERROR_110    L"\nError %d: Failed to detect power source. \n"
#define ID_ERROR_111    L"\nError %d: Display Partition Version failed. \n"
#define ID_ERROR_112    L"\nError %d: Binary is not present. \n"
#define ID_ERROR_113    L"\nError %d: Request and Reply messages' size mismatch. \n"
#define ID_ERROR_114    L"\nError %d: Unsupported Platform: slim SKU is not supported\n"

#define ID_ERROR_115    L"\nError %d: Restore Point Image was requested, but there was Full/Partial FW Update before without Restart after it. \n"
#define ID_ERROR_116    L"\nError %d: Update to incompatible PMC: The PMC instance ID is different, which may be due to H/LP SKU incompatibility.\n"
#define ID_ERROR_117    L"\nError %d: Update to incompatible H/LP SKU image. \n"

#define ID_ERROR_118    L"\nError %d: Failed to communicate with CSME.\nThis tool must be run from a privileged account (administrator/root).\n"
#define ID_ERROR_119    L"\nError %d: Restore Point Image was requested, but there was Full/Partial FW Update before without Restart after it.\n"
#define ID_ERROR_120    L"\nError %d: Update to incompatible PMC: The instance ID is different, which means H/LP incompatibility between the Update Image and the Flash Image.\n"
#define ID_ERROR_121    L"\nError %d: Update Image length is bigger than the expected size of the image according to its size in the flash. For example: Error on updating from Consumer to Corporate.\n"
#define ID_ERROR_122    L"\nError %d: Manifest size in Update Image is bigger than 8KB, or exceeds the Update Image size.\n"
#define ID_ERROR_123    L"\nError %d: Failed to open loader (DEV_FD_LDR_VERIFY_MAN) to verify manifest. \n"
#define ID_ERROR_124    L"\nError %d: Failed to open loader (DEV_FD_LDR_VERIFY_MAN) to install / uninstall keys. \n" 
#define ID_ERROR_125    L"\nError %d: Failed to verify signature of OEM or RoT key manifests.\n"
#define ID_ERROR_126    L"\nError %d: ldr_uninstall_keys() failed - uninstall keys for OEM partitions (ISHC/IUNP). \n"
#define ID_ERROR_127    L"\nError %d: Failed to open loader (DEV_FD_LDR_INSTALL) to install pre-update module. \n"
#define ID_ERROR_128    L"\nError %d: ldr_install_modules() failed - install pre-update module. \n"
#define ID_ERROR_129    L"\nError %d: Failed to create dir or files for preupdate module (permissions, quota). \n"
#define ID_ERROR_130    L"\nError %d: spawn() of pre-update module failed - might be permissions problem. \n"
#define ID_ERROR_131    L"\nError %d: waitpid() to pre-update module failed. \n"
#define ID_ERROR_132    L"\nError %d: Pre-Update module process returned error exit code after it was run. \n"
#define ID_ERROR_133    L"\nError %d: Call to sku_mgr functions failed. \n"
#define ID_ERROR_134    L"\nError %d: Call to cfgmgr functions failed. cfgmgr_get_rule(), cfgmgr_set_rule(). \n"
#define ID_ERROR_135    L"\nError %d: Manifest not found in partition (in Update or Flash Image). \n"
#define ID_ERROR_136    L"\nError %d: Crypto operation (calculating hash of partition) failed. \n"
#define ID_ERROR_137    L"\nError %d: Loader failed to verify manifest signature of FTPR. Production vs. Pre-Production. \n"
#define ID_ERROR_138    L"\nError %d: Loader failed to verify manifest signature of NFTP. \n"
#define ID_ERROR_139    L"\nError %d: Loader failed to verify manifest signature of IDLM. \n"
#define ID_ERROR_140    L"\nError %d: Loader failed to verify manifest signature of RBE. \n"
#define ID_ERROR_141    L"\nError %d: Loader failed to verify manifest signature of PMC. \n"
#define ID_ERROR_142    L"\nError %d: Loader failed to verify manifest signature of OEM KM. \n"
#define ID_ERROR_143    L"\nError %d: Loader failed to verify manifest signature of WCOD. \n"
#define ID_ERROR_144    L"\nError %d: Loader failed to verify manifest signature of LOCL. \n"
#define ID_ERROR_145    L"\nError %d: Loader failed to verify manifest signature of PAVP. \n"
#define ID_ERROR_146    L"\nError %d: Loader failed to verify manifest signature of IOMP. \n"
#define ID_ERROR_147    L"\nError %d: Loader failed to verify manifest signature of MGPP. \n"
#define ID_ERROR_148    L"\nError %d: Loader failed to verify manifest signature of TBTP. \n"
#define ID_ERROR_149    L"\nError %d: Loader failed to verify manifest signature of ISHC. \n"
#define ID_ERROR_150    L"\nError %d: Loader failed to verify manifest signature of IUNIT. \n"
#define ID_ERROR_151    L"\nError %d: Loader failed to verify manifest signature of Pre-Update Module. \n"
#define ID_ERROR_152    L"\nError %d: Some manifest extension is missing in FTPR. \n"
#define ID_ERROR_153    L"\nError %d: Some manifest extension is missing in NFTP. \n"
#define ID_ERROR_154    L"\nError %d: Some manifest extension is missing in IDLM. \n"
#define ID_ERROR_155    L"\nError %d: Some manifest extension is missing in RBE. \n"
#define ID_ERROR_156    L"\nError %d: Some manifest extension is missing in PMC. Wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_157    L"\nError %d: Some manifest extension is missing in OEM KM. Wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_158    L"\nError %d: Some manifest extension is missing in WCOD. \n"
#define ID_ERROR_159    L"\nError %d: Some manifest extension is missing in LOCL. \n"
#define ID_ERROR_160    L"\nError %d: Some manifest extension is missing in PAVP. Wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_161    L"\nError %d: Some manifest extension is missing in IOMP. Wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_162    L"\nError %d: Some manifest extension is missing in MGPP. Wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_163    L"\nError %d: Some manifest extension is missing in TBTP. Wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_164    L"\nError %d: Some manifest extension is missing in ISHC. Wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_165    L"\nError %d: Some manifest extension is missing in IUNIT. Wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_166    L"\nError %d: Some manifest extension is missing in Pre-Update Module. \n"
#define ID_ERROR_167    L"\nError %d: FTPR partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_168    L"\nError %d: NFTP partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_169    L"\nError %d: DLMP partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_170    L"\nError %d: RBEP partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_171    L"\nError %d: PMCP partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_172    L"\nError %d: OEMP partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_173    L"\nError %d: WCOD partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_174    L"\nError %d: LOCL partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_175    L"\nError %d: PAVP partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_176    L"\nError %d: IOMP partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_177    L"\nError %d: MGPP partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_178    L"\nError %d: TBTP partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_179    L"\nError %d: ISHC partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_180    L"\nError %d: IUNP partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_181    L"\nError %d: UPDT partition hash and calculated hash are not the same. If partition hash is zero - wrong MEU Tool was used to create the partition. \n"
#define ID_ERROR_182    L"\nError %d: The size of an Update partition size is bigger than the size of the Flash partition.\n"
#define ID_ERROR_183    L"\nError %d: Location of partition to backup is not inside NFTP.\n"
#define ID_ERROR_184    L"\nError %d: The number of IUPs in the Update/Flash Image is bigger than MAX_IUPS.\n"
#define ID_ERROR_185    L"\nError %d: Partition name inside IUPs list (in FTPR manifest extension) is not IUP.\n"
#define ID_ERROR_186    L"\nError %d: Non-optional IUP (like LOCL, WCOD) inside IUPs list (in FTPR manifest extension) is not in the Update Image.\n"
#define ID_ERROR_187    L"\nError %d: PMC partition is not in the Update Image.\n"
#define ID_ERROR_188    L"\nError %d: It is not allowed to do Partial Update on this partition.\n"
#define ID_ERROR_189    L"\nError %d: It is not allowed to do Partial Update on Type-C partitions, according to NVAR.\n"
#define ID_ERROR_190    L"\nError %d: RBEP and NFTP must have the same version as FTPR, in the Update Image.\n"
#define ID_ERROR_191    L"\nError %d: RBEP and NFTP must have the same SVN as FTPR, in the Update Image.\n"
#define ID_ERROR_192    L"\nError %d: RBEP and NFTP must have the same VCN as FTPR, in the Update Image.\n"
#define ID_ERROR_193    L"\nError %d: Non-optional IUPs (like LOCL, WCOD) must have the same major build version as FTPR, in the Update Image for Full Update, in the Flash Image for Partial Update.\n"
#define ID_ERROR_194    L"\nError %d: Update IUP must not have SVN smaller than SVN of Flash IUP.\n"
#define ID_ERROR_195    L"\nError %d: Update IUP must not have VCN smaller than VCN of Flash IUP.\n"
#define ID_ERROR_196    L"\nError %d: Update Image length is not the same as Flash Image length.\n"
#define ID_ERROR_197    L"\nError %d: Update from PV bit ON to PV bit OFF is not allowed.\n"
#define ID_ERROR_198    L"\nError %d: Update to PV bit OFF on Revenue platform is not allowed.\n"
#define ID_ERROR_199    L"\nError %d: Update to higher SVN must be an upgrade - to higher build version.\n"
#define ID_ERROR_200    L"\nError %d: Update to higher SVN must be to a higher Hot Fix number (the third number in the build version).\n"
#define ID_ERROR_201    L"\nError %d: Place holder. This error code will not be returned by the FW.\n"
#define ID_ERROR_202    L"\nError %d: Place holder. This error code will not be returned by the FW.\n"
#define ID_ERROR_203    L"\nError %d: Place holder. This error code will not be returned by the FW.\n"
#define ID_ERROR_204    L"\nError %d: Place holder. This error code will not be returned by the FW.\n"
#define ID_ERROR_205    L"\nError %d: PMCP must have the same major API version as the version inside the list in FTPR, in the Update Image for Full Update.\n"
#define ID_ERROR_206    L"\nError %d: OEMP must have the same major API version as the version inside the list in FTPR, in the Update Image for Full Update.\n"
#define ID_ERROR_207    L"\nError %d: WCOD must have the same major API version as the version inside the list in FTPR, in the Update Image for Full Update, in the Flash Image for Partial Update.\n"
#define ID_ERROR_208    L"\nError %d: LOCL must have the same major API version as the version inside the list in FTPR, in the Update Image for Full Update, in the Flash Image for Partial Update.\n"
#define ID_ERROR_209    L"\nError %d: PAVP must have the same major API version as the version inside the list in FTPR, in the Update Image for Full Update, in the Flash Image for Partial Update.\n"
#define ID_ERROR_210    L"\nError %d: IOMP must have the same major API version as the version inside the list in FTPR, in the Update Image for Full Update, in the Flash Image for Partial Update.\n"
#define ID_ERROR_211    L"\nError %d: MGPP must have the same major API version as the version inside the list in FTPR, in the Update Image for Full Update, in the Flash Image for Partial Update.\n"
#define ID_ERROR_212    L"\nError %d: TBTP must have the same major API version as the version inside the list in FTPR, in the Update Image for Full Update, in the Flash Image for Partial Update.\n"
#define ID_ERROR_213    L"\nError %d: ISHC must have the same major API version as the version inside the list in FTPR, in the Update Image for Full Update, in the Flash Image for Partial Update.\n"
#define ID_ERROR_214    L"\nError %d: IUNP must have the same major API version as the version inside the list in FTPR, in the Update Image for Full Update, in the Flash Image for Partial Update.\n"
#define ID_ERROR_215    L"\nError %d: Place holder. This error code will not be returned by the FW.\n"
#define ID_ERROR_216    L"\nError %d: Non-optional IUP (like LOCL, WCOD) inside IUPs list (in FTPR manifest extension) is not in the Flash Image.\n"
#define ID_ERROR_217    L"\nError %d: A partition that was searched in the Update Image is not in it.\n"
#define ID_ERROR_218    L"\nError %d: Update between engineering build vs regular build is not allowed. Both builds have to be the same type: regular or engineering build. Engineering build is 7000 and above. Regular build is below 7000.\n"
#define ID_ERROR_219    L"\nError %d: OEM KM partition is not in the Update Image, but ISHC/IUNP is in the Update Image, which is not allowed.\n"
#define ID_ERROR_220    L"\nError %d: ISHC/IUNP do not exist in the same way in the Update Image and in the Flash Image.\n"
#define ID_ERROR_221    L"\nError %d: OEM KM partition is not in the Flash Image, but it is in the Update Image, which is not allowed.\n"
#define ID_ERROR_222    L"\nError %d: Partial FW Update: the Update Image contains IUP that is different than the one that was requested to be updated in the Partial Update command.\n"
#define ID_ERROR_223    L"\nError %d: The Partial Update Image size is different than the size of the IUP in it (as it is in the manifest). This means that the Update Image contains more (or less) than the IUP partition.\n"
#define ID_ERROR_224    L"\nError %d: Bug: Open of IUP path failed. Need to add the path in Storage, or add permissions to FW Update process.\n"
#define ID_ERROR_225    L"\nError %d: Bug: spi_flash_partition_updated() failed. This updates the files (in the file system) of the newly updated IUP, after Partial Update (without reset).\n"
#define ID_ERROR_226    L"\nError %d: Update Rule file contains invalid value. (This file holds the MEBX option for FW Update: values: disable / enable / password protected).\n"
#define ID_ERROR_227    L"\nError %d: Call to pwdmgr function failed. pwdmgr_get_mebx_pwd() - to get the ME password, for FW Update that requires password.\n"
#define ID_ERROR_228    L"\nError %d: Call to pwr function failed. pwr_state_get_last_reset_reason().\n"
#define ID_ERROR_229    L"\nError %d: Call to spi function failed. spi_flash_get_override_strap().\n"
#define ID_ERROR_230    L"\nError %d: Get Restore Point Image is not allowed, because a previous Get Restore Point operation already started. Both operations will be aborted. (Get Restore Point can be started again after this).\n"
#define ID_ERROR_231    L"\nError %d: Bug: Get Restore Point Image Data: The offset of Restore Point Image is bigger than the Image length.\n"
#define ID_ERROR_232    L"\nError %d: Heci message length is not as expected.\n"
#define ID_ERROR_233    L"\nError %d: FWU_START_MSG Heci message contains invalid value in UpdateEnvironment. Value should be FWU_ENV_MANUFACTURING. (Other possible value: FWU_ENV_IFU is obsolete).\n"
#define ID_ERROR_234    L"\nError %d: FWU_DATA Heci command was sent, but the FW Update wasn't started with FWU_START Heci command before it.\n"
#define ID_ERROR_235    L"\nError %d: FWU_DATA Heci command has invalid data length (too big).\n"
#define ID_ERROR_236    L"\nError %d: FWU_END Heci command was sent, but there was no FWU_DATA command before it.\n"
#define ID_ERROR_237    L"\nError %d: Error when flushing NVM to UMA space (before rewriting flash).\n"
#define ID_ERROR_238    L"\nError %d: clear_ipk_valid_bit() returned error. This function prevents CSE from entering M3 after FW Update, and instead CSE will go into MOff.\n"
#define ID_ERROR_239    L"\nError %d: FW Update process called to PG entry override (sys_pg_override()) at the start of the update, and it returned error.\n"
#define ID_ERROR_240    L"\nError %d: Flash Image content is invalid (partitions/manifests sizes, locations, structures). \n"
#define ID_ERROR_241    L"\nError %d: Wrong structure of Update Image (manifests, $CPD), complete_partition_length is 0, no module or metadata of preupdate inside UPDT partition. \n"
#define ID_ERROR_242    L"\nError %d: Base image was provided, but base image doesn't contain independent updatable partitions (IUPs).\n"
#define ID_ERROR_243    L"\nError %d: Provided image is missing IUNP partition.\n"
#define ID_ERROR_244    L"\nError %d: Provided image is missing ISHC partition.\n"
#define ID_ERROR_245    L"\nError %d: Provided image contains IUNP partition but the existing image on the platform doesn't contain it.\n"
#define ID_ERROR_246    L"\nError %d: Provided image contains ISHC partition but the existing image on the platform doesn't contain it.\n"
#define ID_ERROR_247    L"\nError %d: Provided image contains ISHC or/and IUNP partitions but not the OEM Key manifest partition.\n"
#define ID_ERROR_248    L"\nError %d: Provided image contains OEM Key manifest partition but the current platform image don't contain such partition.\n"
#define ID_ERROR_249    L"\nError %d: Display Partition Vendor ID failed.\n"
#define ID_ERROR_250    L"\nError %d: Invalid Partition ID. Use a Partition ID which is on the Flash Image.\n"
#define ID_ERROR_251    L"\nError %d: Update to higher TCB SVN must be also to higher ARB SVN.\n"

// Warning Messages 
#define ID_WARN_0       L"\nWarning: Do not exit the process or power off the machine before the firmware update process ends.\n"

//
// Specify the OEM ID to be passed to the ME
// The bit order must match what was created with FITC.
//
// For example:
// FITC Setting: 00000000-0001-0000-0000-00000000001
// MEINFO.EXE returns: 00000000-0001-0000-0000-00000000001
// 
// mOemId would be: 
//  _UUID                      mOemId = {0x00000000, 0x0001, 0x0000, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01};
//
_UUID                      mOemId = {0x00000000, 0x0000, 0x0000, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

//Defines
// Last Reset Types
#define MFT_PART_INFO_EXT_UPDATE_ACTION_NONE         0
#define MFT_PART_INFO_EXT_UPDATE_ACTION_HOST_RESET   1
#define MFT_PART_INFO_EXT_UPDATE_ACTION_CSE_RESET    2
#define MFT_PART_INFO_EXT_UPDATE_ACTION_GLOBAL_RESET 3

// Get Interface
#define FW_UPDATE_DISABLED 0
#define FW_UPDATE_ENABLED 1
#define FW_UPDATE_PASSWORD_PROTECTED 2

//
//  MSFT compiler uses this variable to inform itself whether floating
//  point support is loaded.
//
int _fltused;

unsigned int      index = 0;
signed int        timer30s = 0;

FlashVersion  mVersion;

//
// Print a message to the desired output device
// Customize for platform
//
void DisplayConsoleMessage (IN CHAR16 *String)
{
    Print(String);
}

// This fucntion prints the progress bar line during hte fw update process. 
static UINT32 ProgressDot(void)
{
    // new code
    INT32 i = 0;
    char string[3][80] = {
        "                       Do not Interrupt", 
        "                       Do not Interrupt", 
        "                       Do not Interrupt"};

    for (i = 0; i < 3; i++)
    {

        Print (L"Progress Dot %s \r", string[i]);
        //SleepInMilliseconds(250);
        //timer30s += 250;
        //timer30s += 250; 
    }
    timer30s += 1;

    return 0;
} 

void DisplaySendStatus(float BytesSent, float BytestobeSent)
{
    float  Value = BytesSent/BytestobeSent * 100;

    UINT32  pValue = (UINT32)Value;

    if (pValue != 100)
    {
        Print (L"Sending the update image to FW for verification:  [ %d%% ]\r",pValue);
    }else 
    {
        Print (L"Sending the update image to FW for verification:  [ COMPLETE ] \n");
    }
}

void DisplayTextForReturnErrorCode(UINT32 Code)
{
    switch(Code) {
    case FWU_NO_MEMORY:
        Print (ID_ERROR_3, FWU_NO_MEMORY);
        break;
    case FWU_UPDATE_TIMEOUT:
        Print (ID_ERROR_4, FWU_UPDATE_TIMEOUT);
        break;
    case FWU_IMG_HEADER:
        Print (ID_ERROR_7, FWU_IMG_HEADER);
        break;
    case FWU_SGN_MISMATCH:
        Print (ID_ERROR_8, FWU_SGN_MISMATCH);
        break;
    case FWU_SKU_MISMATCH: 
        Print (ID_ERROR_9, FWU_SKU_MISMATCH);
        break;
    case FWU_VER_MISMATCH:
        Print (ID_ERROR_10, FWU_VER_MISMATCH);
        break;
    case FWU_USAGE_ERROR:
        Print (ID_ERROR_18, FWU_USAGE_ERROR);
        break;
    case FWU_USAGE_ERROR_B:
        Print (ID_ERROR_18B, FWU_USAGE_ERROR_B);
        break;
    case FWU_LAST_STATUS:
        Print (ID_ERROR_19, FWU_LAST_STATUS);
        break;
    case FWU_GET_VER_ERR:
        Print (ID_ERROR_24 , FWU_GET_VER_ERR);
        break;
    case FWU_AUDIT_POLICY_FAILURE:
        Print (ID_ERROR_25C, FWU_GENERAL);
        break;
    case FWU_ERROR_CREATING_FT:
        Print (ID_ERROR_25C, FWU_GENERAL);
        break;
    case FWU_SAL_NOTIFICATION_ERROR:
        Print (ID_ERROR_25A, FWU_GENERAL);
        break;
    case FWU_NO_UPDATE:
        Print (ID_ERROR_27, FWU_NO_UPDATE);
        break;
    case FWU_LOCAL_DIS:
        Print (ID_ERROR_29, FWU_LOCAL_DIS);
        break;
    case FWU_VERIFY_OEM_ID_ERR:
        Print (ID_ERROR_22 , FWU_VERIFY_OEM_ID_ERR);
        break;
    case FWU_INVALID_OEM_ID:
        Print (ID_ERROR_33, FWU_INVALID_OEM_ID);
        break;
    case FWU_FILE_OPEN:
        Print (ID_ERROR_34, FWU_FILE_OPEN);
        break;
    case FWU_IME_SMALL_BUFF:
        Print (ID_ERROR_36, FWU_IME_SMALL_BUFF);
        break;
    case FWU_IME_NO_DEVICE:
        Print (ID_ERROR_37, FWU_IME_NO_DEVICE);
        break;
    case FWU_IME_NOT_READY:
        Print (ID_ERROR_38, FWU_IME_NOT_READY);
        break;
    case FWU_IME_UN_SUP_MESS:
        Print (ID_ERROR_39, FWU_IME_UN_SUP_MESS);
        break;
    case FWU_INVALID_IMG_LENGTH: 
        Print (ID_ERROR_40, FWU_INVALID_IMG_LENGTH);
        break;
    case FWU_GLBL_BUFF_UNAVAILABLE:
        Print (ID_ERROR_41, FWU_GLBL_BUFF_UNAVAILABLE);
        break;
        //case STATUS_OUT_OF_MEMORY:
        //  Print (ID_ERROR_12, FWU_NO_MEMORY);
        //  break;
    case FWU_INVALID_FW_PARAMS:
        Print (ID_ERROR_44, FWU_INVALID_FW_PARAMS);
        break;
    case FWU_FILE_WRITE:
        Print (ID_ERROR_55, FWU_FILE_WRITE);
        break;
    case FWU_DISPLAY_FW_VERSION:
        Print (ID_ERROR_56, FWU_DISPLAY_FW_VERSION);
        break;
    case FWU_IMAGE_UNDER_VCN:
        Print(ID_ERROR_57, FWU_IMAGE_UNDER_VCN);
        break;
    case FWU_IMAGE_VER_HIST_CHK_FAIL:
        Print (ID_ERROR_58, FWU_IMAGE_VER_HIST_CHK_FAIL);
        break;
    case FWU_DOWNGRADE_VETOED:
        Print (ID_ERROR_59, FWU_DOWNGRADE_VETOED);
        break;
    case FWU_FW_WRITE_FILE_FAIL:
        Print (ID_ERROR_60, FWU_FW_WRITE_FILE_FAIL);
        break;
    case FWU_FW_READ_FILE_FAIL:
        Print (ID_ERROR_61, FWU_FW_READ_FILE_FAIL);
        break;
    case 29:
        Print (ID_ERROR_62, FWU_FW_DELETE_FILE_FAIL);
        break;
    case FWU_PARTITION_LAYOUT_NOT_COMP:
        Print (ID_ERROR_63, FWU_PARTITION_LAYOUT_NOT_COMP);
        break;
    case FWU_DOWNGRADE_NOT_ALLOWED_DATA_MISMATCH:
        Print (L"\nDowngrade is NOT permitted due to mismatched data format version.\n");
        Print (ID_ERROR_64, FWU_DOWNGRADE_NOT_ALLOWED_DATA_MISMATCH);
        break;
    case FWU_UPDATE_PASSWORD_NOT_MATCHED:
        Print (ID_ERROR_65, FWU_UPDATE_PASSWORD_NOT_MATCHED);
        break;
    case FWU_UPDATE_PASSWORD_EXCEED_MAXIMUM_RETRY:
        Print (ID_ERROR_66, FWU_UPDATE_PASSWORD_EXCEED_MAXIMUM_RETRY);
        Print (ID_INFO_16);
        break;
    case FWU_PID_NOT_EXPECTED:
        Print (ID_ERROR_72, FWU_PID_NOT_EXPECTED);
        break;
    case FWU_FILE_ALREADY_EXISTS:
        Print (L"Error %d: File already exists\n", FWU_FILE_ALREADY_EXISTS);
        break;
    case FWU_FILE_INVALID:
        Print (L"Error %d: Invalid File\n", FWU_FILE_INVALID);
        break;
    case FWU_SAVE_RESTORE_POINT_ERROR:
        Print (L"\nError %d: Restore Point Image Failure\n", FWU_SAVE_RESTORE_POINT_ERROR);
        break;
    case FWU_GET_BLIST_ERROR:
        Print (L"\nError %d: Get Black List Failure\n" , FWU_GET_BLIST_ERROR);
        break;
    case FWU_GET_PATTRIB_ERROR:
        Print (L"\nError %d: Get Partition Attribute Failure\n" , FWU_GET_PATTRIB_ERROR);
        break;
    case FWU_GET_UPD_INFO_STATUS:
        Print (L"\nError %d: Get Update Info Status Failure\n" , FWU_GET_UPD_INFO_STATUS);
        break;
    case FWU_BUFFER_COPY_FAILED:
        Print (L"\nError %d:Buffer Copy Failure\n" , FWU_BUFFER_COPY_FAILED);
        break;
    case FWU_ERROR_PMC_INSTANCE:
        Print(ID_ERROR_116, FWU_ERROR_PMC_INSTANCE);
        break;
    case FWU_ERROR_H_LP_MISMATCH:
        Print(ID_ERROR_117, FWU_ERROR_H_LP_MISMATCH);
        break;
    default:
        Print (ID_ERROR_25, FWU_GENERAL);
        break;
    }
}

/*++

Routine Description:

Validate command line is present and remove current working directory if present

Arguments:

ImageHandle - handle for application
CommandLine - buffer of command line arguments passed during Load call

Returns:

GC_TODO: add return values

--*/


BOOLEAN
    ParseLoadOptions (
    EFI_HANDLE  ImageHandle,
    OUT CHAR16  **CommandLine
    )
{
    EFI_LOADED_IMAGE   *Image;
    EFI_STATUS         Status;


    //
    // Set defaults.
    //
    *CommandLine = NULL;

    Status = BS->HandleProtocol (
        ImageHandle,
        &LoadedImageProtocol,
        (VOID *) &Image
        );

    if (!EFI_ERROR (Status)) {
        CHAR16  *CmdLine;

        //
        // Make sure it is power of 2
        //
        UINT32  CmdSize;

        CmdLine = Image->LoadOptions;
        CmdSize = Image->LoadOptionsSize &~1;

        if (CmdLine && CmdSize) {
            //
            // Set command line pointer for caller
            //
            *CommandLine = CmdLine;

            //
            // See if current working directory was passed, if so skip it
            // and move to the first argument
            //
            while((*CmdLine != 0) && CmdSize) {
                CmdLine++;
                CmdSize -= sizeof (CHAR16);
            }

            return TRUE;
        }
    }

    return FALSE;
}

#define WHITESPACE( p ) (InQuote == FALSE && (*p == L' ' || *p == L'\t'))

EFI_STATUS
    ParseCommandLine (
    IN  EFI_HANDLE  ImageHandle,
    IN  CHAR16      *FileName,
    OUT BOOLEAN     *bAllowSV,
    OUT BOOLEAN     *bUsePassword,
    OUT BOOLEAN     *bPid,
    OUT BOOLEAN     *bF,
    OUT BOOLEAN     *bPdt,
    OUT BOOLEAN     *bIshVer
    )
    /*++

    Routine Description:

    Parse the command line

    Arguments:

    ImageHandle - handle for application
    CommandLine - buffer of command line arguments passed during Load call

    Returns:

    GC_TODO: add return values

    --*/
{
    BOOLEAN                   InQuote = FALSE;
    UINT32                    i;
    CHAR16                    *CommandLine;

    *bAllowSV = FALSE;
    *bUsePassword = FALSE;
    *bPid = FALSE;
    *bF   = FALSE;
    *bPdt = FALSE;
    *bIshVer = FALSE;

    if (ParseLoadOptions (ImageHandle, &CommandLine)) {
        //
        // Skip the 1st argument which should be us.
        //

        while (*CommandLine != L' ' && *CommandLine != 0)
        {
            CommandLine++;
        }

        //
        // Get to the beginning of the next argument.
        //
        while (WHITESPACE (CommandLine ))
        {
            CommandLine++;
        }

        if (*CommandLine == 0)
        {
            Print (L"Usage: FwUpdLclApp.efi /p /s /i /d /g <updateimage>\n");
            return EFI_INVALID_PARAMETER;
        }

        FileName[0] = '\0';

        while (*CommandLine)
        {
            //
            // Skip leading white space
            //
            if (WHITESPACE (CommandLine))
            {
                CommandLine += 1;
                continue;
            }

            if ((*CommandLine == '/') || (*CommandLine == '-'))
            {
                CommandLine += 1; // Next char
                //
                // Check char
                //
                switch (*CommandLine)
                {
                case 's':
                    *bAllowSV = TRUE;
                    break;

                case 'p':
                    *bUsePassword = TRUE;
                    break;

                case 'i':
                    *bPid = TRUE;
                    break;

                case 'f':
                    *bF = TRUE;
                    break;

                case 'd':
                    *bPdt = TRUE;
                    break;

                case 'g':
                    *bIshVer = TRUE;
                    break;

                default:
                    Print (L"Usage: FwUpdLclApp.efi /p /s /i /d /g <updateimage>\r\n");
                    return EFI_INVALID_PARAMETER;
                }
                CommandLine += 1; // Next char
            } else 
            {
                // Copy filename
                i = 0;
                while (*CommandLine != 0)
                {
                    FileName[i] = *CommandLine;
                    CommandLine++;
                    i++;
                }

                FileName[i] = '\0';
                break; 
            }
        }

        if (FileName[0] == '\0')
        {
            Print (L"Usage: FwUpdLclApp.efi /p /s /i /d /g <updateimage>\r\n");
            return EFI_INVALID_PARAMETER;
        }
    } else 
    {
        Print (L"Usage: FwUpdLclApp.efi /p /s /i /d /g <updateimage>\r\n");
        return EFI_INVALID_PARAMETER;
    }

    return EFI_SUCCESS;
}

EFI_STATUS
    GetUpdateImage (
    IN  EFI_HANDLE       ImageHandle,
    IN  CHAR16          *ImageName,
    OUT UINTN           *ImageLength,
    IN  UINT8           **ImageBuffer
    )
{
    EFI_STATUS                      Status;
    EFI_FILE_IO_INTERFACE           *Vol;
    EFI_FILE_INFO                   *FileInfo;
    EFI_FILE_HANDLE                 FileHandle;
    EFI_FILE_HANDLE                 RootFs;
    EFI_FILE_HANDLE                 CurDir;
    //UINTN                           ScratchBufferSize;
    unsigned long                            ScratchBufferSize;
    EFI_GUID                        FileInfoGuid = EFI_FILE_INFO_ID;
    EFI_LOADED_IMAGE                *LoadedImage;
    //
    // Open the ME FW Update Image
    //
    //Print(L"\n entered Getupdate Image Func ");
    Status = BS->HandleProtocol (
        ImageHandle,
        &LoadedImageProtocol,
        (VOID *) &LoadedImage
        );

    Status = BS->HandleProtocol (
        LoadedImage->DeviceHandle,
        &FileSystemProtocol,
        (VOID *) &Vol
        );

    Status  = Vol->OpenVolume (Vol, &RootFs);
    CurDir  = RootFs;
    //Print(L"\n entered CurDir open ");
    Status = CurDir->Open (
        CurDir,
        &FileHandle,
        ImageName,
        EFI_FILE_MODE_READ,
        0
        );

    if (EFI_ERROR (Status)) {
        DEBUG ((D_ERROR, "Unable to Open File - %r\n", Status));
        //Print (L"Cannot open update file!\r\n");
        return Status;
    }

    //
    // Figure out how big the file is
    //
    //ScratchBufferSize  = SIZE_OF_EFI_FILE_INFO + 1024;
    ScratchBufferSize  = 1024;
    //Print(L"\n file info scratchbufsize");
    FileInfo = (EFI_FILE_INFO *)AllocatePool (ScratchBufferSize);
    Status = FileHandle->GetInfo (
        FileHandle,
        &FileInfoGuid,
        &ScratchBufferSize,
        FileInfo
        );
    if (EFI_ERROR (Status)) {
        DEBUG ((D_ERROR, "Unable to GetInfo Buffer - %r\n", Status));
        //Print(L"Unable to GetInfo Buffer - %r\n", Status);
        FileHandle->Close (FileHandle);
        FreePool (FileInfo);
        return Status;
    }

    *ImageLength = (UINTN)FileInfo->FileSize;
    //Print("\n Image Length: %d",&ImageLength);
    FreePool (FileInfo);

    //
    // Read the update image into the ImageBuffer
    //
    //Print(L"Enter read update image into buffer ");
    *ImageBuffer = (UINT8 *)AllocatePool (*ImageLength);
    if (!ImageBuffer) {
        DEBUG ((D_ERROR, "Unable to allocate File Buffer - %r\n", Status));
        //Print (L"FwUpdLcl: Cannot allocate memory for Update Image!\r\n");
        Status = EFI_OUT_OF_RESOURCES;
        FileHandle->Close (FileHandle);
        return Status;
    }
    //Print(L"\n enter filehandle->Read");
    Status = FileHandle->Read (
        FileHandle,
        ImageLength,
        (VOID *)*ImageBuffer
        );
    if (EFI_ERROR (Status)) {
        DEBUG ((D_ERROR, "Unable to read file - %r\n", Status));
        //Print (L"FwUpdLcl: Cannot read Update Image into memory!\r\n");
        FileHandle->Close (FileHandle);
        FreePool (ImageBuffer);
        return Status;
    } 

    FileHandle->Close (FileHandle);
    //Print(L"\n Close \n ");

    return Status;
}

char uppercase(char s){

    if( s >=  'a' && s <= 'z')
        return (s = s +'A' - 'a');
    else
        return s;

}

static INT32 stringcompare(const char *s1, const char *s2)
{
    if(NULL == s1 || NULL == s2)
    {
        return 1;
    }

    while (uppercase(*s1) == uppercase(*s2))
    {
        if (*s1 == '\0')
            return 0;
        s1++;
        s2++;
    }
    return uppercase(*s1) - uppercase(*s2);
}

EFI_DRIVER_ENTRY_POINT (InitializeFwUpdLclApplication)

    EFI_STATUS
    InitializeFwUpdLclApplication (
    IN EFI_HANDLE           ImageHandle,
    IN EFI_SYSTEM_TABLE     *SystemTable
    )
{
    EFI_STATUS        Status;
    CHAR16            ImageName[256];
    UINTN               ImageLength = 0;
    UINT8             *ImageBuffer = NULL;
    BOOLEAN           bAllowSV;
    BOOLEAN           bUsePassword;
    BOOLEAN           bPid;
    BOOLEAN           bF;
    BOOLEAN           bPdt;
    BOOLEAN           bIshVer;
    //CHAR              Password[9];
    char              *Password = NULL;
    UINT32            FWUpdateStatus;
    DWORD             loops = 500;
    BOOLEAN           done = FALSE;
    UINT32            lastStatus = 0;
    UINT32            platCheck = 0;
//    FWVersion         fwVersion;
    INT32             platCheckReturn = 0; 
    UINT32            CheckPolicyStatus = 0;
    UPDATE_TYPE       Upd_Type;
    VersionLib        ver;
    UINT32            index = 0;
//    UINT32            status;
    UINT32            UpdateStatus = 0;
    UINT32            TotalStages = 0;
    UINT32            PercentWritten = 0;  
    CHAR8             symbol;
    UINT32            lastResetType;
    UPDATE_FLAGS_LIB  update_flags;
    UINT16            interfaces;
    int               timer30s = 0;
    unsigned int       indexMod;
    int                percentage0s = 0;
    int                percentdiff = 0;
    UINT32             ComparePartID = 0;
    UINT32             hexValueInstId = 0;
    IPU_UPDATED_INFO   IpuUpdatedInfo;
    UINT32             PartId = 0;
    UINT32             fwuError;
    FWU_GET_IPU_PT_ATTRB_MSG_REPLY FwuGetIpuAttrbMsgInfo;
    bool               found = false;
    UINT32             i, j = 0;
    UINT16 major = 0;
    UINT16 minor = 0;
    UINT16 hotfix = 0;
    UINT16 build = 0;
    UINT32 vendorID = 0;
    INT32 itr = 0;
    UINT32 continueLoop;

    //
    // Zero out the update flag structure
    //
    ZeroMem(&update_flags, sizeof(UPDATE_FLAGS_LIB));
    ZeroMem((char*)&IpuUpdatedInfo,sizeof(IPU_UPDATED_INFO));

    //
    // Initialize the EFI Toolkit Library. Set BS, RT, &ST globals
    //  BS = Boot Services RT = RunTime Services
    //  ST = System Table
    //
    InitializeLib (ImageHandle, SystemTable);

    Print (L"\n Intel (R) Firmware Update Utility Sample Application \n");
    //Print (L"\n Intel (R) Firmware Update Utility Version: %d.%d.%d.", TOOLS_VER_MAJOR, TOOLS_VER_MINOR, TOOLS_VER_HOTFIX);
    //Print (L"%d\n", TOOLS_VER_BUILD);

    Print (L"\n");

    Print (ID_INFO_1);

    //
    // Determine the command line arguments
    //
    Status = ParseCommandLine (ImageHandle, ImageName, &bAllowSV, &bUsePassword, &bPid, &bF, &bPdt, &bIshVer);
    if (EFI_ERROR (Status)) 
    {
        DEBUG ((D_ERROR, "Unable to process command line - %r\n", Status));
        return Status;
    }

    //
    // Display ISH FW Version with '/g' option
    //
    if (bIshVer){
        FWUpdateStatus = GetPartVersion(FPT_PARTITION_NAME_ISHC, &major, &minor, &hotfix, &build);
        if (FWUpdateStatus != FWU_ERROR_SUCCESS)
        {
            DEBUG ((D_ERROR, "GetPartVersion Error %r\n", FWUpdateStatus));
            return EFI_SUCCESS;
        }
        Print(L"ISH FW Version: %d.%d.%d.%d \n\n", major, minor, hotfix, build);

        Status = GetPartVendorID(FPT_PARTITION_NAME_ISHC, &vendorID);
        if (FWUpdateStatus != FWU_ERROR_SUCCESS)
        {
            Print(L"GetPartVendorID Error %d\n", FWUpdateStatus);
            return EFI_SUCCESS;
        }
        Print(L"ISH FW Vendor ID: 0x%x \n\n", vendorID);

        return EFI_SUCCESS;
    }

    //
    // Load image into memory buffer
    //
    Print (L"\n Loading image into memory : ...\n");

    Status = GetUpdateImage (ImageHandle, ImageName, &ImageLength, &ImageBuffer);
    if (EFI_ERROR (Status)) {
        Print(L" %r ",Status);
        return Status;
    }

    if (bPdt)
    {
        Print(L"Sending Image for Executing PDT Update. \n");

        Status = HeciPdt((char *)ImageBuffer, (unsigned int)ImageLength);
        if (EFI_ERROR(Status)) {
            Print(L"Send Failed. \n");
        }
        else {
            Print(L"Send Succeeded. \n");
        }
    }
    else
    {
        //
        // Get the current status of the ME FWUpdate Client - verifies if the client is 
        // installed
        //

        if (GetLastStatus(&lastStatus))
        {
            Print (ID_ERROR_19, FWU_LAST_STATUS);
            return EFI_SUCCESS;
        }
        //
        // Is there a pending reset?
        //
        /* switch (lastStatus)
        {
        case STATUS_UPDATE_IMG_LOADING: //fall through
        case STATUS_UPDATE_IMG_AUTHENTICATING: //fall through
        case STATUS_UPDATE_IMG_PROCESSING: //fall through
        case STATUS_UPDATE_CREATING_FT: //fall through
        case STATUS_UPDATE_UPDATING_CODE: //fall through
        case STATUS_UPDATE_UPDATING_NFT:
            Print(ID_ERROR_20, FWU_UPD_PROCESS);
            //g_fwuError = FWU_UPD_PROCESS;
            //progExit(g_fwuError);
            break;
        }*/

        if (GetLastUpdateResetType (&lastResetType)) 
        {
            Print (ID_ERROR_19, FWU_LAST_STATUS);
            return EFI_SUCCESS;
        }
        if (STATUS_UPDATE_HOST_RESET_REQUIRED == lastStatus)
        {
            Print (ID_ERROR_51, FWU_REBOOT_NEEDED);
            return EFI_SUCCESS;
        }

        if (IsUpdateStatusPending (lastStatus))
        {
            Print (ID_ERROR_20, FWU_UPD_PROCESS);
            return EFI_SUCCESS;
        }

        switch (lastResetType) 
        {
        case MFT_PART_INFO_EXT_UPDATE_ACTION_HOST_RESET:

        case MFT_PART_INFO_EXT_UPDATE_ACTION_GLOBAL_RESET:
            Print (ID_ERROR_51, FWU_REBOOT_NEEDED);
            return EFI_SUCCESS;
            break;
        default:

            break;
        }


        Print (ID_INFO_3);

        //
        // Is update supported? 
        //
        if (GetInterfaces (&interfaces))
        {
            Print (ID_ERROR_19, FWU_LAST_STATUS);
            return EFI_SUCCESS;
        }

        switch (interfaces) 
        {
        case FW_UPDATE_DISABLED:    
            Print (L"Local FWUpdate is Disabled\n"); 
            return EFI_SUCCESS;
        case FW_UPDATE_PASSWORD_PROTECTED:
            Print (L"Local FWUpdate is Password Protected\n"); 
            break;
        case FW_UPDATE_ENABLED:
            break;
        default:
            break;
        }

        Print (L"\n Checking Firmware Parameters ... \n \n");
        CheckPolicyStatus = CheckPolicyBuffer((char *)ImageBuffer, (INT32)ImageLength, (INT32)bAllowSV, &Upd_Type, &ver);

        switch (Upd_Type) 
        {
        case DOWNGRADE_SUCCESS:

        case SAMEVERSION_SUCCESS:

        case UPGRADE_SUCCESS:
            break;

        case DOWNGRADE_FAILURE:
            Print (L"FW Update downgrade not allowed\n");
            return EFI_SUCCESS;
            break;
        case SAMEVERSION_FAILURE:
            Print (L"FW Update same version not allowed, specify /s on command line\n");
            return EFI_SUCCESS;
            break;
        default:
            break;
        }

        if(bPid)
        {
            Print(L"\n Executing ISH Partial FWUpdate");

            ComparePartID = FPT_PARTITION_NAME_ISHC; 
            //Print(L"\n compareID: 0x%x",ComparePartID);

            //Get Partition Attribute from Firmware
            fwuError = GetExtendedIpuPartitionAttributes(&FwuGetIpuAttrbMsgInfo, FWU_IPU_UPDATE_OPERATION);
            if (FWU_ERROR_SUCCESS != fwuError)
            {
                DisplayTextForReturnErrorCode(fwuError);
                return fwuError;
            }

            PartId = 0;
            //Loop through expected partitions from FW to find partition requested
            for(j=0;j<FwuGetIpuAttrbMsgInfo.NumOfPartition;j++)
            {
                if(ComparePartID == FwuGetIpuAttrbMsgInfo.PtAttribute[j].PtNameId)
                {
                    PartId = FwuGetIpuAttrbMsgInfo.PtAttribute[j].PtNameId;
                    found = true;
                    break;
                }
            }

            if(!found)
            {
                DisplayTextForReturnErrorCode(FWU_PID_NOT_EXPECTED);
                //Print(L"ParID: 0x%x\tInstId: 0x%x \n",ComparePartID,hexValueInstId);
                return FWU_PID_NOT_EXPECTED;
            }
            Print(L"%s", ID_WARN_0);
            ////Actual Partial FW update
            //
            // Password hack for testing - replace with OEM version if password required
            // 
            if (!bUsePassword) 
            {
                ZeroMem (Password, sizeof (Password));
            }
            if (bUsePassword)
            {
                FWUpdateStatus = FwUpdatePartialBuffer ((char *)ImageBuffer, (unsigned int)ImageLength,PartId,0,&IpuUpdatedInfo,"P@ssw0rd",FWU_ENV_MANUFACTURING, mOemId, update_flags, &DisplaySendStatus);
            } 
            else
            { 
                FWUpdateStatus = FwUpdatePartialBuffer ((char *)ImageBuffer, (unsigned int)ImageLength,PartId,0,&IpuUpdatedInfo,Password, FWU_ENV_MANUFACTURING, mOemId, update_flags, &DisplaySendStatus);
            }

            if (FWU_ERROR_SUCCESS != FWUpdateStatus)
            {
                DisplayTextForReturnErrorCode(FWUpdateStatus);
                if (ImageBuffer)
                {
                    FreePool (ImageBuffer);
                }
                return EFI_SUCCESS;
            }

            if (ImageBuffer) 
            {
                FreePool (ImageBuffer);
            }
        }else{
            //
            // Password hack for testing - replace with OEM version if password required
            // 
            Print(L"\n");
            Print(L"%s \n", ID_WARN_0);
            /*if (!bUsePassword) 
            {
                ZeroMem (Password, sizeof (Password));
            }*/

            //if (bUsePassword)
            //{
            //  FWUpdateStatus = FwUpdateFullBuffer ((char *)ImageBuffer, (unsigned int)ImageLength, "P@ssw0rd", 0, FWU_ENV_MANUFACTURING, mOemId, update_flags, &DisplaySendStatus);
            //} 
            //else
            //{
            FWUpdateStatus = FwUpdateFullBuffer ((char *)ImageBuffer, (unsigned int)ImageLength, Password, 0, FWU_ENV_MANUFACTURING, mOemId, update_flags, &DisplaySendStatus);
            //}

            if (FWU_ERROR_SUCCESS != FWUpdateStatus)
            {
                DisplayTextForReturnErrorCode(FWUpdateStatus);
                if (ImageBuffer)
                {
                    FreePool (ImageBuffer);
                }
                return EFI_SUCCESS;
                //return FWUpdateStatus;
            }

            if (ImageBuffer) 
            {
                FreePool (ImageBuffer);
            }
        }

        // 
        // Image downloaded to FW Update Client
        // Now query the status of the update being applied
        //
        Print(L"\n FW Update:  [  0 %% ]\r");

        index  =  0;
        //
        // Loop through Polling for Fw Update Stages
        //
        ProgressDot();
        do {
            //We mod4 the index to determine which ascii animation frame to display for this iteration.
            indexMod = (++index % 4);
            //symbol = (++index % 2 == 0)?'|': '-';
            switch(indexMod)
                //loop through (|) (/) (-) (\) (|) (/) ...
            {
            case CMD_LINE_STATUS_UPDATE_1: symbol =  '|';  break;
            case CMD_LINE_STATUS_UPDATE_2: symbol =  '/';  break;
            case CMD_LINE_STATUS_UPDATE_3: symbol =  '-';  break;
            case CMD_LINE_STATUS_UPDATE_4: symbol =  '\\'; break;
            }

            //Loop to retry 3 times when there is a Heci Send or Receive failure
            itr = 0;
            do
            {
                //Loop to retry 3 times when there is a Heci Send or Receive failure
                Status = FWUpdate_QueryStatus_Get_Response(&UpdateStatus, &TotalStages,
                    &PercentWritten, &lastStatus, &lastResetType);

                // *** TODO check status before checking PercentWritten 
                //don't print Fw Update percent if its more than 100% (its probably error value)
                if (PercentWritten > 100)
                {
                    break;
                }

                Print (L"FW Update:  [ %d%% (%c)]\r" ,PercentWritten, symbol);

                itr++;
            } while (Status == FWU_IME_NOT_READY && itr < 3);


            BS->Stall(250000); // Wait 250 milliseconds before polling again
            if (timer30s >= 30) 
            {
                percentdiff = PercentWritten - percentage0s;
                if (percentdiff == 0)
                {
                    // *** TODO: check this comment
                    //TODO: Add timeout when add cmdline option
                    // Status = FWU_UPDATE_TIMEOUT;
                } else 
                {
                    percentage0s = PercentWritten;
                    timer30s = 0;
                }
            } else 
            {
                timer30s++;
            }

            if (UpdateStatus != HECI1_CSE_GS1_PHASE_FWUPDATE)// we are NOT in FW update phase
            {
                if (lastStatus == STATUS_UPDATE_NOT_READY)// waiting for HECI message -> continue to wait
                    continueLoop = 1;
                else 
                    continueLoop = 0;
            }
            else //we are in FW update phase
            {
                if (PercentWritten == 100) //FWupdate proccess completed successfully -> exit
                    continueLoop = 0;
                else 
                {
                    if (lastStatus == STATUS_UPDATE_SUCCESS) //FWupdate proccess in progress (0-99%) -> continue loop
                        continueLoop = 1;
                    else // we got error from FW update (progress) registers e.g. 127 (0x7F) and not a value between 0-99%
                    {
                        if (lastStatus == STATUS_UPDATE_NOT_READY) // waiting for HECI message -> continue to wait
                            continueLoop = 1;
                        else
                            continueLoop = 0;
                    }
                }
            }
        } while (continueLoop);

        switch (Status)
        {
        case FWU_NO_MEMORY:

        case FWU_IME_NO_DEVICE:
            Print (ID_ERROR_68, FWU_UPDATE_POLLING_FAILED);
            return EFI_SUCCESS;
        case FWU_IME_NOT_READY:
            DisplayTextForReturnErrorCode(Status);
            return EFI_SUCCESS;
        case FWU_ERROR_FW:
            Print (ID_ERROR_69, FWU_ERROR_FW, UpdateStatus);
            return EFI_SUCCESS;
        default:
            break;
        }

        switch (lastStatus) 
        {
        case STATUS_SUCCESS:
            //some time we exit the progress loop corectly but the "PercentWritten" is still 99 so just for customers output consistency we disply 100%
            if(PercentWritten == 99)
            {
                Print("L\nFW Update:  [  100 %% ]\r");
            }

            switch (lastResetType) 
            {
            case MFT_PART_INFO_EXT_UPDATE_ACTION_NONE:

            case MFT_PART_INFO_EXT_UPDATE_ACTION_CSE_RESET:
                Print (L"\nFW Update is completed successfully.\n");
                break;
            case MFT_PART_INFO_EXT_UPDATE_ACTION_HOST_RESET:

            case MFT_PART_INFO_EXT_UPDATE_ACTION_GLOBAL_RESET:
                Print (L"\nFW Update is complete and a reboot will run the new FW.\n");
                break;
            default:
                Print (L"\nFW Update is complete and a reboot will run the new FW.\n");
                break;
            }
            fwuError = FWU_ERROR_SUCCESS;
            break;
        case STATUS_UPDATE_IMAGE_INVALID:
            DisplayTextForReturnErrorCode(FWU_IMG_HEADER);
            break;
        case STATUS_UPDATE_INTEGRITY_FAILURE:
            DisplayTextForReturnErrorCode(FWU_SGN_MISMATCH);
            break;
        case STATUS_UPDATE_SKU_MISMATCH:
            DisplayTextForReturnErrorCode(FWU_SKU_MISMATCH);
            break;
        case STATUS_UPDATE_FW_VERSION_MISMATCH:
            DisplayTextForReturnErrorCode(FWU_VER_MISMATCH);
            break;
        case STATUS_UPDATE_GENERAL_FAILURE:
            DisplayTextForReturnErrorCode(FWU_GENERAL);
            break;
        case STATUS_UPDATE_OUT_OF_RESOURCES:
            DisplayTextForReturnErrorCode(FWU_NO_MEMORY);
            break;
        case STATUS_UPDATE_AUDIT_POLICY_FAILURE:
            DisplayTextForReturnErrorCode(FWU_AUDIT_POLICY_FAILURE);
            break;
        case STATUS_UPDATE_ERROR_CREATING_FT:
            DisplayTextForReturnErrorCode(FWU_ERROR_CREATING_FT);
            break;
        case STATUS_UPDATE_SAL_NOTIFICATION_ERROR:
            DisplayTextForReturnErrorCode(FWU_SAL_NOTIFICATION_ERROR);
            break;
        case STATUS_INVALID_OEM_ID:
            DisplayTextForReturnErrorCode(FWU_INVALID_OEM_ID);
            break;
        case STATUS_DOWNGRADE_NOT_ALLOWED_VCN_RESTRICTION:
            DisplayTextForReturnErrorCode(FWU_IMAGE_UNDER_VCN);
            break; 
        case STATUS_DOWNGRADE_NOT_ALLOWED_SVN_RESTRICTION: 
            Print("FW downgrade is not allowed due to SVN restriction.\n");
            break;
        case STATUS_UPDATE_IMAGE_BLACKLISTED:
            Print("FW update/downgrade is not allowed to the supplied FW image.\n");
            break;
        default:
            DEBUG ((D_ERROR, "lastStatus = %d\n",lastStatus));
            DisplayTextForReturnErrorCode(FWU_GENERAL);
            break;
        }
    }
    return EFI_SUCCESS;
}
